/*
 * Bouncing Ball, a simple example for an Android app made with
 * Processing, 27.03.2012 Andreas Freise
 * http://www.gwoptics.org
 *
 * This app has been tested with Processing 2.05a, Android SDK 17
 * on the following devices HTC Nexus 1, Sony Tablet S
 *
 * Read http://wiki.processing.org/w/Android for the official info
 * on Android development with Processing and then the very good 
 * step-by-step description on exporting the app by AK Eric 
 * http://www.akeric.com/blog/?p=1352
 *
 * Note this sketch requires the following two lines in the 
 * AndroidManifest.xml file when exporting:
 * android:screenOrientation="landscape"
 * android:configChanges="orientation|keyboardHidden">
 * in order to work on some devices, e.g. the HTC Nexus 1
 *
 * This is open software, provided under the GNU Public License v3.
 */

/* import android configuration to check screen layout of device */
import android.content.res.Configuration;
boolean rotate_axes;

/* Ketai is an open source (GPLv3) library that offers an easy access 
 * to the sesnors of the device. Version 5 is used here to read the 
 * accelerometers.
 * http://danielsauter.com/display.php?project_id=113
 * http://code.google.com/p/ketai/
 */
import ketai.sensors.*;
KetaiSensor sensor;
float accelerometerX, accelerometerY;

// some global variables to simplify matters
boolean showinfo; // show info text?
boolean followpointer;  // ball follows pointer?
float fps=60; // frames per second for frameRate() command
Ball ball;
int state; // interger to switch between loading and run
PGraphics bg;  // background image

void setup()
{
  size(screenWidth, screenHeight, P3D); // use OPENGL for speed!

  // Find out whether we are running a device with long or wide screen
  // (this is necessary for the screen X/Y axes to align with the accelerometers)
  // We keep the screen orientation fixed ...
  orientation(LANDSCAPE);
  // ... but have to rotate accelerometer readout
  Configuration config = getResources().getConfiguration();
  if ((config.screenLayout & Configuration.SCREENLAYOUT_LONG_MASK) == Configuration.SCREENLAYOUT_LONG_YES) 
  {
    println("screen is long (portrait)");
    rotate_axes=true;
  } 
  else {
    println("screen is wide (landscape)");
    rotate_axes=false;
  }

  frameRate(fps);  // set framerate explicitly

  // init global variables
  followpointer=false;
  showinfo=false;
  state = 0; // we start in the `loading' state

  bg = createGraphics(screenWidth, screenHeight, P2D);

  // for text output we might need to set some text properties
  textAlign(CENTER, TOP);
  textSize(24);
} 

void draw()
{
  switch (state) { // using states to do image loading in draw
  case 0: 
    // just display 'loading' text
    background(0);
    pushStyle();
    fill(200);
    text("Loading ... ", width/2.0, 30);
    popStyle();
    state=1;
    break;
  case 1:
    // do image loading and other setup functions...
    ball = new Ball(width/2.0, height/2.0); // start Ball in the middle of the screen
    ball.init();

    println("Screen width: "+width+", height: "+height);
    PImage load_bg=loadImage("background_800x480px.png");

    // rescale and crop image to screen size
    if (load_bg!=null) { 
      float scale1=width/(float)load_bg.width;
      float scale2=height/(float)load_bg.height;
      float imgscale = Math.max(scale1, scale2);
      println("resizing image by "+imgscale);
      load_bg.resize(ceil(load_bg.width*imgscale), ceil(load_bg.height*imgscale));
      println("image size is now " + load_bg.width+" x " +load_bg.height);
      bg.beginDraw();
      bg.image(load_bg, 0, 0);
      bg.endDraw();
      state=2;
    }
    break;
  default:
    /* this should work but does not
     hint(DISABLE_DEPTH_TEST);
     background(bg);
     hint(ENABLE_DEPTH_TEST);
     */
    // *must* use a background command or the sketch is very slow!
    background(0); 
    pushStyle();
    imageMode(CORNER);
    image(bg, 0, 0);
    popStyle();

    if (followpointer) {
      ball.reset(mouseX, mouseY, 0, 0); // set ball position to finger
    }
    else {
      if (ball.isMoving ()) {
        ball.move();
      }
    }
    ball.display();

    if (showinfo) {
      pushStyle();
      fill(200);
      text("Bouncing Ball: \n" + 
        "A simple example app made with Processing\n" +
        "See http://www.gwoptics.org for more!", screenWidth/2.0, 30);
      popStyle();
    }
    // if the phone has been shaken, wake up the ball
    if (accelerometerX*accelerometerX+accelerometerY* accelerometerY>12*12) {
      ball.move();
    }
  }
}

/*
 * Read the touch events to simply know when a finger is
 * on the screen or not.
 */
public boolean surfaceTouchEvent(MotionEvent me) {
  final int action = me.getAction() & MotionEvent.ACTION_MASK;
  if (action == MotionEvent.ACTION_DOWN || action == MotionEvent.ACTION_POINTER_DOWN)
  {
    followpointer=true;
    ball.reset(mouseX, mouseY, 0, 0); // set ball position to finger
    // overwrite previous mouse position so that ball drops vertically
    pmouseX=mouseX;
    pmouseY=mouseY;
  }
  else if (action == MotionEvent.ACTION_UP || action == MotionEvent.ACTION_POINTER_UP) 
  {
    followpointer=false;
    ball.reset(mouseX, mouseY, 2*(mouseX-pmouseX), 2*(mouseY-pmouseY)); // throw ball!
  }
  return super.surfaceTouchEvent(me);
}

/* Read accelerometer data and set two gloabal variables */
void onAccelerometerEvent(float x, float y, float z)
{
  if (rotate_axes) { // for screens with long screen layout
    accelerometerX = -1.0 * y;
    accelerometerY = x;
  }
  else { // for screens with wide screen layout
    accelerometerX = x;
    accelerometerY = y;
  }
}

/* Check for MENU button */
void keyPressed() {
  if (key == CODED) {
    if (keyCode == MENU) {
      if (showinfo) {
        showinfo=false;
      }
      else {
        showinfo=true;
      }
      keyCode=1;
    }
  }
}

/* Override the parent (super) Activity class:
 * States onCreate(), onStart(), and onStop() aren't called by the sketch.  
 * Processing is entered at onResume() and exits at onPause()
 */
public void onResume() {
  super.onResume(); 
  sensor = new KetaiSensor(this); // init sensors
  sensor.start(); // start sensors
}  
public void onPause() {
  sensor.stop(); // stop sensors when skecht is paussed/quit
  super.onPause();
}

