/**
 *  Copyright notice
 *  
 *  This file is part of the Processing sketch `GBViewer' 
 *  http://www.gwoptics.org/processing/gbviewer
 *  
 *  Copyright (C) 2009 onwards Daniel Brown and Andreas Freise
 *  
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  as published by the Free Software Foundation; either version 2
 *  of the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 *  MA  02110-1301, USA.
 */


import org.gwoptics.Logo;
import org.gwoptics.LogoSize;
import org.gwoptics.graphics.GWColour;

import guicomponents.GButton;
import guicomponents.GCheckbox;
import guicomponents.GComponent;
import guicomponents.GConstants;
import guicomponents.GFont;
import guicomponents.GLabel;
import guicomponents.GSlider;
import guicomponents.GWSlider;
import guicomponents.GOption;
import guicomponents.GOptionGroup;

public class GBViewer extends PApplet {
  Logo _logo;
  GraphControl _grph;
  GWSlider sldN,sldM;
  GWSlider sldWaistDist;
  GWSlider sldXYRange;
  GWSlider sldZRange;
  GLabel lblXMode, lblYMode;
  GLabel lblWaistDist;
  GLabel lblXAxis, lblZAxis;
  GLabel lblWaistDistUnit;
  GLabel lblXAxisUnit, lblZAxisUnit;
  GLabel lblCameraBtns;
  GOption opGauss,opLag,opSinLag;
  GOptionGroup opGroupPlots;
  GCheckbox chkAutoRange;
  GButton btnTop,btnBottom, btnForward, btnSide, btnProjection, btnReset;

  private GBViewerSettings[] _settings;
  private boolean _updateGraphSettings = true;
  PImage _background;
  PImage _hg_eq, _lg_cos_eq, _lg_hel_eq;

  public void setup() {
    try{				  
      size(900,600);
      //setLayout(new BorderLayout());

      _background = loadImage("data/drawing.png");
      _hg_eq = loadImage("hg_eq.png");
      _lg_cos_eq = loadImage("lg_cos_eq.png");
      _lg_hel_eq = loadImage("lg_hel_eq.png");

      _grph = new GraphControl();
      _grph.init();
      setLayout(null); 
      add(_grph); 

      opGroupPlots = new GOptionGroup();
      opGauss = new GOption(this,"Hermite-Gauss Beam",625, 196,200);
      opLag = new GOption(this,"Helical Laguerre-Gauss Beam",625, 216,200);
      opSinLag = new GOption(this,"Sinusoidal Laguerre-Gauss Beam",625, 236,200);
      opGroupPlots.addOption(opGauss);
      opGroupPlots.addOption(opLag);
      opGroupPlots.addOption(opSinLag);
      opGroupPlots.setSelected(0);

      int nmslidersYPos = 280;	  
      int slidersYPos = 300;	  

      btnTop = new GButton(this,"Top",0,430,120,20);
      btnBottom = new GButton(this,"Bottom",120,430,120,20);
      btnForward = new GButton(this,"Forward",240,430,120,20);
      btnSide = new GButton(this,"Side",360,430,120,20);
      btnProjection = new GButton(this,"Orthographic View",480,430,120,20);

      
      GWColour c1,c2,c3;
      c1 = new GWColour(220/255f,220/255f,220/255f);
      c2 = new GWColour(230/255f,230/255f,230/255f);
      c3 = new GWColour(200/255f,200/255f,200/255f);
      
      int ic1=c1.toInt();
      int ic2=c2.toInt();
      int ic3=c3.toInt();
      
      btnTop.setColours(ic1,ic2,ic3);
      btnBottom.setColours(ic1,ic2,ic3);
      btnForward.setColours(ic1,ic2,ic3);
      btnSide.setColours(ic1,ic2,ic3);
      btnProjection.setColours(ic1,ic2,ic3);

      lblXMode = new GLabel(this,"n mode:",615,nmslidersYPos - 3,50);
      lblYMode = new GLabel(this,"m mode:",615,nmslidersYPos - 3 + 50,50);
      lblWaistDist     = new GLabel(this,"axis pos(z):",615,slidersYPos - 6 + 100,80); 
      lblWaistDistUnit = new GLabel(this,"[m]",615,slidersYPos + 9 + 100,80); 

      sldN = new GWSlider(this,"blue18pxOdd",700,nmslidersYPos,180);	  
      sldM = new GWSlider(this,"blue18pxOdd",700,nmslidersYPos + 50,180);
      sldN.setValueType(GWSlider.INTEGER);
      sldM.setValueType(GWSlider.INTEGER);
      
      sldN.setStickToTicks(true);
      sldM.setStickToTicks(true);	 
      
      sldN.setRenderValueLabel(false);
      sldM.setRenderValueLabel(false);  

      sldWaistDist = new GWSlider(this,"blue18pxOdd",700,slidersYPos + 100,180);		
      sldWaistDist.setValueType(GWSlider.DECIMAL);  
      sldWaistDist.setPrecision(1);
      sldWaistDist.setRenderValueLabel(false);
      //sldWaistDist.unit = "[m]";

      lblXAxis = new GLabel(this,"x/y range:",615,slidersYPos + 144,100);
      lblXAxisUnit = new GLabel(this,"[m]",615,slidersYPos + 159,100);
      sldXYRange = new GWSlider(this,"blue18pxOdd",700,slidersYPos + 150,180);
      sldXYRange.setValueType(GWSlider.EXPONENT);
      sldXYRange.setPrecision(1);
      sldXYRange.setRenderValueLabel(false);
      //sldXYRange.unit = "[m]";

      lblZAxis = new GLabel(this,"z range [a.u.]:",615,slidersYPos + 191,100);
      //lblZAxisUnit = new GLabel(this,"[a.u.]",615,slidersYPos + 194,100);
      sldZRange = new GWSlider(this,"blue18pxOdd",700,slidersYPos + 200,180);
      sldZRange.setValueType(GWSlider.INTEGER);
      sldZRange.setPrecision(1);
      sldZRange.setRenderValueLabel(false);
      //sldZRange.unit = "[V/m]";

      chkAutoRange = new GCheckbox(this,"auto",617,slidersYPos + 208,110);
      chkAutoRange.setSelected(true);
      _grph.setAutoRange(chkAutoRange.isSelected());

      btnReset = new GButton(this,"reset",615,550,40,17);
      btnReset.setColours(ic1,ic2,ic3);

      _logo = new Logo(this, 770,570,true,LogoSize.Size25);

      //load up default settings from enums
      _settings = new GBViewerSettings[GraphOptions.values().length];
      GraphOptions[] ops = GraphOptions.values();

      for (int i = 0; i < ops.length; i++) {
        _settings[i] = ops[i].getDefaultSettings();			
      }

      this.registerMouseEvent(this);
      this.registerDispose(this);

      _loadSettings(GraphOptions.GUASS_AMP);
      _grph.setPlot(GraphOptions.GUASS_AMP);
      _refreshGraphSettings();
      _refreshGraphPlot();
      noLoop();

    }
    catch(Exception e){
      PApplet.println("Exception occured during setup phase.");
      PApplet.println(e.getMessage());
      PApplet.println(e.getStackTrace());
      super.stop();
    }
  }

  public void dispose(){
    //Make sure embedded PApplet disposes correctly
    _grph.stop();
  }

  public void mouseEvent(MouseEvent e){
    if(e.getID() == MouseEvent.MOUSE_CLICKED || e.getID() == MouseEvent.MOUSE_DRAGGED || e.getID() == MouseEvent.MOUSE_RELEASED)
      _grph.redraw();

    this.redraw();
  }

  public void draw() {
    try{
      pushStyle();
      image(_background,0,0);

      pushMatrix();
      fill(0);
      textAlign(LEFT);

      textFont(GFont.getFont(this, "Arial Bold", 17)); 
      text("Gaussian Beam Viewer",620,10,250,400);
      textFont(GFont.getFont(this, "Arial", 11)); 
      text("v1.0 (build 0511090249)",615,575,250,400);
      textFont(GFont.getFont(this, "Arial", 11)); 

      String s = "" +
        "This applet allows you to experiment with various forms of Gaussian laser beams. You " +
        "can alter the mode of the beam along with the distance from the beam waist with the sliders below. The 3D graph is a cross section of the beam and represents " +
        "its amplitude. By dragging with the left mouse button over the graph you can change the camera position, and zoom in/out with the right button.";

      text(s, 620, 42, 260, 500);
      popMatrix();	
      popStyle();

      pushStyle();
      noFill();
      stroke(50);

      fill(0);
      text("Shown below is the equation of the spatial mode, only the amplitude (shown in blue) is plotted.", 3, 465, 550, 500);

      switch (opGroupPlots.selectedIndex()) {
      case 0:		
        rect(24,489,551,_hg_eq.height + 1);
        image(_hg_eq,25,490);
        break;
      case 1:	
        rect(24,489,551,_lg_hel_eq.height + 1);
        image(_lg_hel_eq,25,490);
        break;
      case 2:	
        rect(24,489,551,_lg_cos_eq.height + 1);
        image(_lg_cos_eq,25,490);
        break;
      }

      popStyle();

      _logo.draw();
    }
    catch(Exception e){
      PApplet.println("Exception occured during draw phase.");
      PApplet.println(e.getMessage());
      PApplet.println(e.getStackTrace());
      super.stop();
    }
  }

  private void _applySettings(GraphOptions newOp, GraphOptions oldOp){		
    //_saveSettings(oldOp);
    //We want to persist settings across options rather than remembering 
    //individual settings for each
    _saveSettings(newOp);
    _loadSettings(newOp);
  }

  private void _saveSettings(GraphOptions op){
    _settings[op.ordinal()].nModeValue = sldN.getValue();

    // removed, 05/11/09 Andreas
    //if(sldM.getValue() > sldN.getValue() && (op ==GraphOptions.LAG_GAUSS_AMP || op == GraphOptions.SINU_LAG_AMP))
    //	_settings[op.ordinal()].mModeValue = sldN.getValue();
    //else

      _settings[op.ordinal()].mModeValue = sldM.getValue();

    _settings[op.ordinal()].waistDistanceValue = sldWaistDist.getValuef();
    _settings[op.ordinal()].XYRangeValue = sldXYRange.getValuef();
    _settings[op.ordinal()].ZRangeValue = sldZRange.getValue();
  }

  private void _loadSettings(GraphOptions op){
    GBViewerSettings s  = _settings[op.ordinal()];

    _updateGraphSettings = false;

    lblXMode.setText(s.nModeLabel);
    lblYMode.setText(s.mModeLabel);

    sldN.setLimits(s.nModeValue,0,s.nModeMax);
    sldN.setTickCount(s.nModeMax);

    sldM.setLimits(s.mModeValue,0,s.mModeMax);
    sldM.setTickCount(s.mModeMax);

    String[] nmlabels={
      "0","1","2","3","4","5","6","7","8","9","10"    };
    sldN.setTickLabels(nmlabels);
    sldM.setTickLabels(nmlabels);

    sldWaistDist.setLimits(s.waistDistanceValue, 0, s.waistDistanceMax);
    sldXYRange.setLimits(s.XYRangeValue, s.XYRangeMin, s.XYRangeMax);
    sldZRange.setLimits(s.ZRangeValue, s.ZRangeMin, s.ZRangeMax);

    _updateGraphSettings = true;

    this.redraw();
  }

  private void _refreshGraphSettings(){
    _grph.g3d.setXAxisMin(-sldXYRange.getValuef());
    _grph.g3d.setXAxisMax(sldXYRange.getValuef());
    _grph.g3d.setYAxisMin(-sldXYRange.getValuef());
    _grph.g3d.setYAxisMax(sldXYRange.getValuef());

    // ignore zRange if auto is selected
    if(chkAutoRange.isSelected()) {
      sldZRange.setValue(_grph.g3d.getZAxisMax());
    }
    else {
      _grph.g3d.setZAxisMin(-sldZRange.getValuef());
      _grph.g3d.setZAxisMax(sldZRange.getValuef());
    }

    int n = sldN.getValue();
    int m = sldM.getValue();
    float wd = sldWaistDist.getValuef();

    _grph.alterEquation(n,m,wd);
  }

  private void _refreshGraphPlot(){
    //bit of a anti-OOP way of doing this but nevermind...
    _grph.g3d.plotSurfaceTrace(0);
  }

  public void handleCheckboxEvents(GCheckbox checkbox) {
    //if(!chkAutoRange.isSelected()){
    //	_refreshGraphSettings();
    //}

    _grph.setAutoRange(chkAutoRange.isSelected());
    _refreshGraphSettings();
    _refreshGraphPlot();
  }

  public void handleOptionEvents(GOption selected, GOption deselected){
    GraphOptions desel = GraphOptions.values()[opGroupPlots.deselectedIndex()];

    switch (opGroupPlots.selectedIndex()) {
    case 0:		
      _applySettings(GraphOptions.GUASS_AMP, desel);
      _grph.setPlot(GraphOptions.GUASS_AMP);
      break;
    case 1:	
      _applySettings(GraphOptions.LAG_GAUSS_AMP, desel);
      _grph.setPlot(GraphOptions.LAG_GAUSS_AMP);
      break;
    case 2:	
      _applySettings(GraphOptions.SINU_LAG_AMP, desel);
      _grph.setPlot(GraphOptions.SINU_LAG_AMP);
      break;
    }

    _refreshGraphSettings();
    _refreshGraphPlot();
  }

  public void handleSliderEvents(GSlider slider) {
    // ignore zRnage if auto is selected
    // Daniel - Changed so that clicking the slider
    // when auto is on disables it and goes to manual		
    
    if(slider == sldZRange && chkAutoRange.isSelected() && slider.eventType == GConstants.CHANGED) {
      _updateGraphSettings = false;			
      sldZRange.setValue(_grph.g3d.getZAxisMax());
      _updateGraphSettings = true;
    }
    else if(slider == sldZRange && chkAutoRange.isSelected() && slider.eventType == GConstants.RELEASED){
      chkAutoRange.setSelected(false);
      _grph.setAutoRange(chkAutoRange.isSelected());
    }					

    if(_updateGraphSettings){
      _refreshGraphSettings();
      _refreshGraphPlot();		
    }
  }	

  public void handleButtonEvents(GButton button) {
    if(button.eventType == GComponent.CLICKED){
      if(button == btnTop)
        if(_grph.isOrthographicView())
          _grph.cam.setPosition(new PVector(400,840,399));
        else
          _grph.cam.setPosition(new PVector(400,700,399));
      else if(button == btnBottom)
        if(_grph.isOrthographicView())
          _grph.cam.setPosition(new PVector(400,-440,399));
        else
          _grph.cam.setPosition(new PVector(400,-700,399));
      else if(button == btnSide)
        if(_grph.isOrthographicView())
          _grph.cam.setPosition(new PVector(-160,200,400));
        else
          _grph.cam.setPosition(new PVector(-500,200,400));
      else if(button == btnForward)
        if(_grph.isOrthographicView())
          _grph.cam.setPosition(new PVector(400,200,-160));
        else
          _grph.cam.setPosition(new PVector(400,200,-500));
      else if(button == btnProjection){
        _grph.toggleViewMode();

        if(_grph.isOrthographicView())
          btnProjection.setText("Perspective View");
        else
          btnProjection.setText("Orthographic View");

      }
      else if(button == btnReset){
        opGroupPlots.setSelected(0);
        _loadSettings(GraphOptions.GUASS_AMP);
        _grph.setPlot(GraphOptions.GUASS_AMP);

        sldN.setValue(0);
        sldM.setValue(0);

        if(_grph.isOrthographicView())
          _grph.toggleViewMode();

        btnProjection.setText("Orthographic View");
        //_settings[opGroupPlots.selectedIndex()] = GraphOptions.values()[opGroupPlots.selectedIndex()].getDefaultSettings();
        //_loadSettings(GraphOptions.values()[opGroupPlots.selectedIndex()]);
        chkAutoRange.setSelected(true);
        _grph.setAutoRange(chkAutoRange.isSelected());
        _refreshGraphSettings();
        _refreshGraphPlot();
        _grph.resetView();

      }

    }		
    _grph.draw();
  }

}

