/**
 *  Copyright notice
 *  
 *  This file is part of the Processing sketch `GBViewer' 
 *  http://www.gwoptics.org/processing/gbviewer
 *  
 *  Copyright (C) 2009 onwards Daniel Brown and Andreas Freise
 *  
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  as published by the Free Software Foundation; either version 2
 *  of the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 *  MA  02110-1301, USA.
 */
 
import java.awt.event.MouseEvent;

import org.gwoptics.ValueType;
import org.gwoptics.gaussbeams.GaussMode;
import org.gwoptics.graphics.camera.Camera3D;
import org.gwoptics.graphics.colourmap.presets.FlipColourmap;
import org.gwoptics.graphics.graph3D.*;

import processing.core.PApplet;
import processing.core.PVector;


@SuppressWarnings("serial")
public class GraphControl extends PApplet {
	
	Camera3D cam;
	public SurfaceGraph3D g3d;
	private int _xtraFrameCount=-20; 
	private int _xtraFrameLimit = 5;
	private GraphOptions _currPlot;
	private boolean isOrtho;
	private boolean _autoRange = false;
	
	public boolean isOrthographicView(){return isOrtho;}
	
	public void setAutoRange(boolean value){
		_autoRange = value;
		
		if(value)
			g3d.setAutoRanging(0);
		else
			g3d.setAutoRanging(-1);

		g3d.plotSurfaceTrace(0);
	}
	
	class BeamEquation{	
		private float lambda = 1064e-9f;
		private float q = 1.0e-3f;
		private float z = 0.0f;
		private float nr = 1.0f;
		
		GaussMode gauss;
		
		public BeamEquation(){
			gauss = new GaussMode(lambda, q, z, nr);
		}
			
		public void setWaistDistance(float value){
			if(z != value){
				z = value;
				gauss = new GaussMode(lambda, q, z, nr);
			}	
		}
	}
	
	class GuassAmpEq extends BeamEquation implements IGraph3DCallback{	   
	  int n = 1;
	  int m = 1;
	  
	  public float computePoint(float X, float Y) {
	    return (float) (gauss.u_nm_amp(n,m,X * 1e-2,Y * 1e-2));
	  }
	}
	
	class LagGuassAmpEq extends BeamEquation implements IGraph3DCallback{
	  int p = 1;
	  int l = 0;
	  
	  public float computePoint(float X, float Y) {
	    return (float) (gauss.u_pl_amp_xy(p,l,X * 1e-2,Y * 1e-2));
	  }
	}

	class SinuLagGuassAmpEq extends BeamEquation implements IGraph3DCallback{
		  int p = 1;
		  int l = 0;
		  
		  public float computePoint(float X, float Y) {
		    return (float) (gauss.u_pl_cos_amp_xy(p,l,X * 1e-2,Y * 1e-2));
		  }
		}
	
	private GuassAmpEq _gAeq = new GuassAmpEq();
	private LagGuassAmpEq _lgAeq = new LagGuassAmpEq();
	private SinuLagGuassAmpEq _slgAeq = new SinuLagGuassAmpEq();
	
	public void setup() {
	  size(600, 430, P3D); //Use P3D for now, openGl seems to have some issues
	  cam = new Camera3D(this);
	  resetView();
	  //constructor arguments are:
	  //(PApplet parent, float xLength, float yLength, float zLength)
	  g3d = new SurfaceGraph3D(this, 800, 800,400,true);				
	  g3d.setXAxisMin(-0.5f);		
	  g3d.setXAxisMax(0.5f);
	  g3d.setZAxisMin(-600);
	  g3d.setZAxisMax(600);		
	  g3d.setYAxisMin(-0.5f);		
	  g3d.setYAxisMax(0.5f);
	  g3d.setXAxisLabelAccuracy(3);
	  g3d.setYAxisLabelAccuracy(3);
	  g3d.setZAxisLabelAccuracy(1);
	  g3d.setXAxisLabelType(ValueType.DECIMAL);
	  g3d.setYAxisLabelType(ValueType.DECIMAL);
	  g3d.setZAxisLabelType(ValueType.DECIMAL);
	  g3d.setXAxisLabel("X Distance from optical axis [cm]");
	  g3d.setYAxisLabel("Y Distance from optical axis [cm]");
	  g3d.setZAxisLabel("Amplitude [a.u.]");
	  g3d.setBillboarding(true); 
	  _currPlot = GraphOptions.GUASS_AMP;	
	  g3d.addSurfaceTrace(_gAeq, 100, 100, new FlipColourmap(true));
	  g3d.plotSurfaceTrace(0);  
	  
	  this.registerMouseEvent(this);
	 
	  //noLoop();
	}
	
	public void resetView(){
		cam.setLookat(new PVector(400,200,400));
		cam.setPosition(new PVector(400,400,-700));
	}

	public void toggleViewMode(){
		if(!isOrtho){
			cam.setOrthographicView();
			isOrtho = true;
		}else{
			cam.setPerspectiveView();
			isOrtho = false;
		}
	}

	public void redraw(){
		_xtraFrameCount = 0;
		loop();
		super.redraw();
	}
	
	public void draw() {  
	  background(170,175,190);
	  g3d.draw();
	  //PVector camPos=cam.getPosition();
          //println(camPos);
	  _xtraFrameCount += 1;
	  
	  if(_xtraFrameCount >= _xtraFrameLimit){
		  noLoop();
	  }
	}

	public void mouseEvent(MouseEvent event){
		if(event.getID() == MouseEvent.MOUSE_DRAGGED){
			_xtraFrameCount = 0;
			loop();
		}
	}
	
	public void setPlot(GraphOptions op){		
		switch (op) {
		case GUASS_AMP:
			if(_currPlot != GraphOptions.GUASS_AMP){
				g3d.removeSurfaceTrace(0);
				g3d.addSurfaceTrace(_gAeq, 120, 120, new FlipColourmap(true));
			}
			break;
			
		case LAG_GAUSS_AMP:
			if(_currPlot != GraphOptions.LAG_GAUSS_AMP){
				g3d.removeSurfaceTrace(0);
				g3d.addSurfaceTrace(_lgAeq, 120, 120, new FlipColourmap(true));
			}
			break;
			
		case SINU_LAG_AMP:
			if(_currPlot != GraphOptions.SINU_LAG_AMP){
				g3d.removeSurfaceTrace(0);
				g3d.addSurfaceTrace(_slgAeq, 120, 120, new FlipColourmap(true));
			}
			break;
		}		
		
		if(_autoRange)
			g3d.setAutoRanging(0);
		
		_currPlot = op;
	}
	
	public void alterEquation(int n,int m, float waistDist){
		int p,l;
		p = n;
		l = m;
		 
		// remoed, 05/11/09 Andreas
		//if(l > p)
		//	l = p;
		
		switch (_currPlot) {
		case GUASS_AMP:
			_gAeq.n = n;
			_gAeq.m = m;
			_gAeq.setWaistDistance(waistDist);
			break;
			
		case LAG_GAUSS_AMP:
			_lgAeq.p = p;
			_lgAeq.l = l;
			_lgAeq.setWaistDistance(waistDist);			
			break;
			
		case SINU_LAG_AMP:			
			_slgAeq.p = p;
			_slgAeq.l = l;
			_slgAeq.setWaistDistance(waistDist);		
			break;	
		}		
	}
}
